#!/bin/bash

# Synchronise FTP Server with Aegis Monita Command Server
#
# This batch file syncs the following directories:
#    command/out -> ftp/out
#    ftp/in      -> command/in
#
# The batch file also deletes any unwanted files in these directories, preserving only files of
# the forms "out/123456.cmd", "in/123456.ok" and "in/123456.fail", and only as long as they have the 
# correct internal format.
#
# This batch file should be called every few minutes.
#
# This batch file requires a small C utility program: moncheck.exe.  This program checks the validity
# of an Aegis Monita file, and can compare the internal timestamps of two valid files.

# Common variables
ROOTPATH="/home/logger"
INPATH="/home/logger/in"
OUTPATH="/home/logger/out"
MONCHECK="/root/moncheck/moncheck"
LOGFILE="/var/log/moncheck.log"
RSYNCPASS="/root/moncheck/rsync.secrets"

# Declare functions required

# lab00 - Log Message for when only one of the files .ok .fail .cmd are kept
lab00()
{
	#echo "Lab00 used, script exiting"
	continue
}

# lab7 - Deleting of CMD Files when OK/Fail file found
lab7() 
{
	echo "New OK/FAIL file found - deleting $1" >> $LOGFILE
	rm -rf $1
} 

# lab6 - Log message for shutdown of processing
lab6 () {
	#echo "Lab6 break loop"
	continue
}

# lab5 - delete .ok files when .fail is also present
lab5() 
{
	# $1 = okfile, $2 = failfile, $3 = cmdfile
	echo "OK and FAIL files both present - deleting $1" >> $LOGFILE
	rm -rf $1
}

# lab4 -  If both CMD and OK/FAIL files exist, delete the older.  If they have the same time, delete CMD
lab4() 
{
	# $1 = okfile, $2 = failfile, $3 = cmdfile
	if [ -e $1 ];
	then
		#Set OK file to RSP
		rspfile="$1"
	else
		#Set FAIL file to RSP
		rspfile="$2"
	fi
	if [ ! -e $3 ];
	then
		# No CMD file, goto Lab6
		lab6
	elif [ ! -e $rspfile ];
	then
		# No OK or Fail files, goto Lab6
		lab6
	else
		/root/moncheck/moncheck -q $3 $rspfile
		result="$?"
		if [ $result -eq 3 ];
		then
			lab6
		elif [ $result -eq 2 ];
		then
			lab7 $3
		elif [ $result -eq 1 ];
		then
			echo "New CMD file found - deleting $rspfile" >> $LOGFILE
			rm -rf $rspfile
			lab6
		else
			lab7 $3
		fi
	fi
}

# Validity Checking
checkfile1 () 
{
	echo "Validity Checking $1" >> $LOGFILE
	if [ -e $1 ] ;
	then
		# Check file against moncheck
		/root/moncheck/moncheck -q $1
		if [ $? -lt 4 ] ;
	 	then
			# File is valid, break loop
			lab00
		else	
			# Bad file
			echo "Invalid filename format - deleting $1"  >> $LOGFILE
			rm -rf $1
		fi
	else
		# Delete files of bad format
		rm -rf $1*
		lab00
	fi
}

# Response Checking
checkfile2 () 
{
	echo "Response-Checking $1" >> $LOGFILE
	okfile="$INPATH/$1.ok"
	failfile="$INPATH/$1.fail"
	cmdfile="$OUTPATH/$1.cmd"
	# If both OK and Fail files exists, delete the older 
	if [ ! -e "$okfile" ] ;
	then
		# No OKfile, goto Lab4
		lab4 $okfile $failfile $cmdfile
	elif [ ! -e "$failfile" ];
	then
		# No FAILfile, goto Lab4
		lab4 $okfile $failfile $cmdfile
	else
		/root/moncheck/moncheck -q $okfile $failfile
		check2res="$?"
		if [ $check2res -eq 3 ];
		then	
			# Other error from Moncheck, goto lab4
			lab4 $okfile $failfile $cmdfile
		elif [ $check2res -eq 2 ];
		then
			# Moncheck gave 2, okfile is older
			lab5 $okfile $failfile $cmdfile
		else
			# Delete Failfile as both OK and Fail are present
			echo "OK and FAIL files both present - deleting $failfile" >> $LOGFILE
			rm -rf $failfile
			lab4 $okfile $failfile $cmdfile
		fi	
	fi
}

# Run script

# Some logging for time purposes

echo "=============================" >> $LOGFILE
echo " Moncheck run at `date +%c`" >> $LOGFILE
echo "=============================" >> $LOGFILE

# For loop to check each file in OUT folder and call checkfile1
for outfile in `ls $OUTPATH` ;
do
fullout="$OUTPATH/$outfile"
# Run checkfile1 over this file
checkfile1 $fullout
done

# Run Rsync for both IN and OUT folders
# Rsync command - OUT
/usr/bin/rsync -az --password-file=`echo $RSYNCPASS` --delete `echo $OUTPATH` remotebackup@192.168.4.6::logger >> $LOGFILE
# Rsync command - IN
/usr/bin/rsync -az --password-file=`echo $RSYNCPASS` --remove-source-files remotebackup@192.168.4.6::logger/in `echo $ROOTPATH` >> $LOGFILE

# For loop to check each file in IN folder and call checkfile1

for infile in `ls $INPATH` ;
do
	fullin="$INPATH/$infile"
	# Run checkfile1 over this file
	checkfile1 $fullin
done

# For loop to compare files in IN/OUT folders for deletion

for bothfile in `ls $INPATH` `ls $OUTPATH` ;
do
	# Extract number from filename
	numfile=`echo $bothfile | cut -d. -f1`
	checkfile2 $numfile
done
# Exit



