/**********************************************************************************
*
*   moncheck - compare the internal timestamps of two Monita FTP files and
*              set the error level to indicate which is newer
*
*   Usage: moncheck [-q] file1 [file2]
*       -q = quiet mode
*       rc=0 - internal timestamps are the same (or only one file is specified and it is OK)
*       rc=1 - file1 is newer
*       rc=2 - file2 is newer
*       rc=4 - one or both files are not valid Monita files
*       rc=3 - other error
*
*   - Compiled for Windows using Borland C.
*   - Compiled for Linux using cc and gcc.
*
**********************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define BYTE_COUNT      33
#define BUFFER_SIZE     200000

unsigned char buf1[ BUFFER_SIZE ];
unsigned char buf2[ BUFFER_SIZE ];
int quiet = 0;

void usage( void )
{
    printf( "Usage: moncheck [-q] file1 [file2]\n" );
    printf( "Where: -q = quiet mode\n" );
    printf( "Returns rc=1 if file1 is newer based on internal timestamps\n" );
    printf( "     or rc=2 if file2 is newer based on internal timestamps\n" );
    printf( "     or rc=0 if they are the same (or only one file is specified)\n" );
    printf( "     or rc=4 if the file(s) are not valid Monita files\n" );
    printf( "     or rc=3 if there is any other problem\n" );
    exit( 3 );
}

FILE *openfile( char *name )
{
    FILE *fp;
    int i;
    char *s;
    int ok = 1;

    // Strip off any quotes enclosing the filename
    if( *name == '"' )
        name++;
    if( ( i = strlen( name ) ) > 0 && name[ i - 1 ] == '"' )
        name[ i - 1 ] = '\0';

    // Serial numbers are of two possible forms: "999999" or  "99999A999".
    // Only accept valid filenames: ...in\12345A678.ok, ...in\12345A678.fail, ...out\12345A678.cmd
    // Directory names and suffixes must be lower case, since Unix uses a case-sensitive filesystem.
    for( i = strlen( name ); i > 0; i-- )
        if( name[ i - 1 ] == '\\' || name[ i - 1 ] == '/' )
            break;

    if( i >= 3 && ( !memcmp( name + i - 3, "in/", 3 ) || !memcmp( name + i - 3 , "in\\", 3 ) ) )
    {
        s = name + i;
        for( i = 0; i < 10; i++ )
            if( !( isdigit( s[ i ] ) || ( isalpha( s[ i ] ) && i == 5 ) ) )
                break;
        if( ( i != 6 && i != 9 ) || ( strcmp( s + i, ".ok" ) && strcmp( s + i, ".fail" ) ) )
            ok = 0;
    }
    else if( i >= 4 && ( !memcmp( name + i - 4, "out/", 4 ) || !memcmp( name + i - 4 , "out\\", 4 ) ) )
    {
        s = name + i;
        for( i = 0; i < 10; i++ )
            if( !( isdigit( s[ i ] ) || ( isalpha( s[ i ] ) && i == 5 ) ) )
                break;
        if( ( i != 6 && i != 9 ) || strcmp( s + i, ".cmd" ) )
            ok = 0;
    }
    else
        ok = 0;

    if( !ok )
    {
        if( !quiet ) printf( "moncheck: file '%s' is not of form: in/12345A678.ok, in/12345A678.fail or out/12345A678.cmd\n", name );
        exit( 4 );
    }

    // Finally, try to open the file...
    if( ( fp = fopen( name, "rb" ) ) == NULL )
    {
        if( !quiet ) printf( "moncheck: Cannot open file '%s' for reading\n", name );
        exit( 3 );
    }

    return fp;
}

void readfile( unsigned char *buf, FILE *fp, char *name )
{
    int n;

    /*
    *   Expected file structure: 
    *       "#" NN NN
    *           01 0c 00 "MONITA123456" or 01 0f 00 "MONITA12345A678" 
    *           02 06 00 <6-byte-timestamp>
    *           ...
    *           04 03 00 "EOF"
    */

    if( ( n = fread( buf, 1, BUFFER_SIZE, fp ) ) < BYTE_COUNT 
     || buf[0] != '#'
     || ( memcmp( &buf[3],  "\x01\x0c\0MONITA", 9 ) && memcmp( &buf[3],  "\x01\x0f\0MONITA", 9 ) )
     || ( memcmp( &buf[18], "\x02\x06\0", 3 ) && memcmp( &buf[21], "\x02\x06\0", 3 ) )
     || memcmp( &buf[n-6], "\x04\x03\0EOF", 6 ) )
    {
        if( !quiet ) printf( "moncheck: File '%s' is not a valid Monita command or response file\n", name );
        exit( 4 );
    }
}

int main( int argc, char **argv )
{
    FILE *fp1, *fp2;
    int rc;

    // Process the options
    while( argc >= 2 && *argv[1] == '-' )
    {
        if( argv[1][1] == 'q' || argv[1][1] == 'Q' )
            quiet = 1;
        else
            usage();
        argc--;
        argv++;
    }

    if( argc < 2 )
        usage();

    // Open the files
    fp1 = openfile( argv[1] );

    if( argc < 3 )
        fp2 = NULL;
    else
        fp2 = openfile( argv[2] );

    // Read and validate the files
    readfile( buf1, fp1, argv[1] );
    fclose( fp1 );

    if( fp2 != NULL )
    {
        readfile( buf2, fp2, argv[2] );
        fclose( fp2 );

        // Compare the timestamps
        rc = memcmp( &buf1[21], &buf2[21], 6 );
        if( rc > 0 )
        {
            if( !quiet ) printf( "1\n" );
            exit( 1 );
        }
        else if( rc < 0 )
        {
            if( !quiet ) printf( "2\n" );
            exit( 2 );
        }
    }

    if( !quiet ) printf( "0\n" );
    exit( 0 );

    return 0;
}
